from datetime import datetime, timedelta
import random
from .database import get_db_connection
from passlib.context import CryptContext

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

class UserModel:
    @staticmethod
    def create_user(email: str, password: str, full_name: str) -> dict:
        conn = get_db_connection()
        cursor = conn.cursor(dictionary=True)
        
        try:
            hashed_password = pwd_context.hash(password)
            cursor.execute(
                """
                INSERT INTO users (email, password_hash, full_name)
                VALUES (%s, %s, %s)
                RETURNING id, email, full_name, is_active, is_admin
                """,
                (email, hashed_password, full_name)
            )
            user = cursor.fetchone()
            conn.commit()
            return user
        finally:
            cursor.close()
            conn.close()

    @staticmethod
    def get_user_by_email(email: str) -> dict:
        conn = get_db_connection()
        cursor = conn.cursor(dictionary=True)
        
        try:
            cursor.execute("SELECT * FROM users WHERE email = %s", (email,))
            return cursor.fetchone()
        finally:
            cursor.close()
            conn.close()

    @staticmethod
    def create_verification_code(user_id: int, code_type: str = "2fa") -> str:
        conn = get_db_connection()
        cursor = conn.cursor()
        
        try:
            code = ''.join(random.choices('0123456789', k=6))
            expires_at = datetime.utcnow() + timedelta(minutes=5)
            
            cursor.execute(
                """
                INSERT INTO verification_codes (user_id, code, type, expires_at)
                VALUES (%s, %s, %s, %s)
                """,
                (user_id, code, code_type, expires_at)
            )
            conn.commit()
            return code
        finally:
            cursor.close()
            conn.close()

    @staticmethod
    def verify_code(user_id: int, code: str, code_type: str = "2fa") -> bool:
        conn = get_db_connection()
        cursor = conn.cursor()
        
        try:
            cursor.execute(
                """
                SELECT * FROM verification_codes
                WHERE user_id = %s AND code = %s AND type = %s AND expires_at > NOW()
                ORDER BY created_at DESC LIMIT 1
                """,
                (user_id, code, code_type)
            )
            result = cursor.fetchone()
            return bool(result)
        finally:
            cursor.close()
            conn.close()